/*
 *  File: ViewstateSaver.java 
 *  Copyright (c) 2004-2007  Peter Kliem (Peter.Kliem@jaret.de)
 *  A commercial license is available, see http://www.jaret.de.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 */
package de.jaret.util.ui.table.addon;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.ui.IMemento;

import de.jaret.util.ui.table.JaretTable;
import de.jaret.util.ui.table.model.IColumn;
import de.jaret.util.ui.table.model.IHierarchicalJaretTableModel;
import de.jaret.util.ui.table.model.IHierarchicalTableViewState;
import de.jaret.util.ui.table.model.IRow;
import de.jaret.util.ui.table.model.ITableNode;
import de.jaret.util.ui.table.model.ITableViewState;

/**
 * Implementation of an IViewstateSaver. This implementation saves and restores
 * <ul>
 * <li>column position and visibility</li>
 * <li>row height and height mode</li>
 * <li>node visibility in case of a hierarchical model</li>
 * <li>autofilter enablement</li>
 * </ul>
 * It does <b>not</b> alter
 * <ul>
 * <li>fixed columns and rows</li>
 * </ul>
 * This viewstate saver can be used as a singleton and is imlemented as one.
 * 
 * @author Peter Kliem
 * @version $Id: ViewstateSaver.java 354 2007-04-09 13:53:45Z olk $
 */
public class ViewstateSaver implements IViewstateSaver {
    /** singleton instance. */
    private static ViewstateSaver _instance = new ViewstateSaver();
    /** key for the columns. */
    private static final String COLUMNS = "columns";
    /** key for the rows. */
    private static final String ROWS = "rows";
    /** key for the nodes. */
    private static final String NODES = "nodes";
    /** key for the autofilter. */
    private static final String AUTOFILTERENABLED = "autofilterenabled";

    /**
     * Get the singleton instance.
     * 
     * @return singleton instance
     */
    public static ViewstateSaver getInstance() {
        return _instance;
    }

    IMemento createOrGetChild(IMemento memento, String name) {
        IMemento m = memento.getChild(name);
        if (m == null) {
            m = memento.createChild(name);
        }
        return m;
    }

    /**
     * {@inheritDoc}
     */
    public void saveViewstate(JaretTable jaretTable, String additionalId, IMemento memento) {
        if (additionalId != null) {
            memento = createOrGetChild(memento, additionalId);
        }
        if (jaretTable.getHierarchicalModel() != null) {
            // hierarchical model
            IMemento nodeMemento = createOrGetChild(memento, NODES);
            saveNodeStates(jaretTable, nodeMemento);
        }
        IMemento colMemento = createOrGetChild(memento, COLUMNS);
        saveColumnState(jaretTable, colMemento);
        IMemento rowMemento = createOrGetChild(memento, ROWS);
        saveRowState(jaretTable, rowMemento);

        // autofilter
        memento.putInteger(AUTOFILTERENABLED, jaretTable.getAutoFilterEnable() ? 1:0);
        
    }

    /**
     * {@inheritDoc}
     */
    public void restoreViewstate(JaretTable jaretTable, String additionalId, IMemento memento) {
        if (additionalId != null) {
            memento = createOrGetChild(memento, additionalId);
        }
        IMemento nodeMemento = memento.getChild(NODES);
        if (nodeMemento != null) {
            readNodeState(jaretTable, nodeMemento);
        }

        IMemento colMemento = memento.getChild(COLUMNS);
        if (colMemento != null) {
            readColumnState(jaretTable, colMemento);
        }

        IMemento rowMemento = memento.getChild(ROWS);
        if (rowMemento != null) {
            readRowState(jaretTable, rowMemento);
        }

        // autofilter
        Integer i = memento.getInteger(AUTOFILTERENABLED);
        boolean afenable = i != null && i.intValue()==1;
        jaretTable.setAutoFilterEnable(afenable);
        
    }

    private void saveNodeStates(JaretTable jaretTable, IMemento nodeMemento) {
        IHierarchicalTableViewState viewstate = (IHierarchicalTableViewState) jaretTable.getTableViewState();
        IHierarchicalJaretTableModel model = jaretTable.getHierarchicalModel();
        ITableNode root = model.getRootNode();
        saveNodeState(viewstate, root, nodeMemento);
    }

    private void saveNodeState(IHierarchicalTableViewState viewstate, ITableNode node, IMemento nodeMemento) {
        nodeMemento.putInteger("expanded" + node.getId(), viewstate.isExpanded(node) ? 1 : 0);
        for (ITableNode n : node.getChildren()) {
            saveNodeState(viewstate, n, nodeMemento);
        }
    }

    private void readNodeState(JaretTable jaretTable, IMemento nodeMemento) {
        IHierarchicalTableViewState viewstate = (IHierarchicalTableViewState) jaretTable.getTableViewState();
        IHierarchicalJaretTableModel model = jaretTable.getHierarchicalModel();
        ITableNode root = model.getRootNode();
        readNodeState(viewstate, root, nodeMemento);
    }

    private void readNodeState(IHierarchicalTableViewState viewstate, ITableNode node, IMemento nodeMemento) {
        Integer state = nodeMemento.getInteger("expanded" + node.getId());
        if (state != null) {
            boolean expanded = state.intValue() == 1;
            viewstate.setExpanded(node, expanded);
        }
        for (ITableNode n : node.getChildren()) {
            readNodeState(viewstate, n, nodeMemento);
        }
    }

    private void saveRowState(JaretTable jaretTable, IMemento rowMemento) {
        ITableViewState viewstate = jaretTable.getTableViewState();
        for (int i = 0; i < jaretTable.getTableModel().getRowCount(); i++) {
            IRow row = jaretTable.getTableModel().getRow(i);
            ITableViewState.RowHeightMode heightMode = viewstate.getRowHeigthMode(row);
            rowMemento.putString("heightMode" + row.getId(), heightMode.name());
            int height = viewstate.getRowHeight(row);
            rowMemento.putInteger("height" + row.getId(), height);
        }
    }

    private void readRowState(JaretTable jaretTable, IMemento rowMemento) {
        ITableViewState viewstate = jaretTable.getTableViewState();
        for (int i = 0; i < jaretTable.getTableModel().getRowCount(); i++) {
            IRow row = jaretTable.getTableModel().getRow(i);
            String modeString = rowMemento.getString("heightMode" + row.getId());
            if (modeString != null) {
                ITableViewState.RowHeightMode heightMode = ITableViewState.RowHeightMode.valueOf(modeString);
                viewstate.setRowHeightMode(heightMode);
                if (heightMode != ITableViewState.RowHeightMode.OPTIMAL) {
                    Integer height = rowMemento.getInteger("height" + row.getId());
                    if (height != null) {
                        viewstate.setRowHeight(row, height.intValue());
                    }
                }
            }
        }
    }

    private void saveColumnState(JaretTable jaretTable, IMemento colMemento) {
        ITableViewState viewstate = jaretTable.getTableViewState();
        for (int i = 0; i < jaretTable.getTableModel().getColumnCount(); i++) {
            IColumn col = jaretTable.getTableModel().getColumn(i);
            int width = viewstate.getColumnWidth(col);
            colMemento.putInteger("width" + col.getId(), width);
            colMemento.putInteger("visible" + col.getId(), viewstate.getColumnVisible(col) ? 1 : 0);
            colMemento.putInteger("pos" + col.getId(), viewstate.getSortedColumns().indexOf(col));
        }
    }

    private void readColumnState(JaretTable jaretTable, IMemento colMemento) {
        Map<Integer, IColumn> colsOrdered = new HashMap<Integer, IColumn>();

        ITableViewState viewstate = jaretTable.getTableViewState();
        for (int i = 0; i < jaretTable.getTableModel().getColumnCount(); i++) {
            IColumn col = jaretTable.getTableModel().getColumn(i);
            Integer width = colMemento.getInteger("width" + col.getId());
            if (width != null) {
                viewstate.setColumnWidth(col, width.intValue());
            }
            Integer visible = colMemento.getInteger("visible" + col.getId());
            if (visible != null) {
                viewstate.setColumnVisible(col, visible.intValue() == 1);
            }
            Integer pos = colMemento.getInteger("pos" + col.getId());
            if (pos != null && pos.intValue() != -1) {
                colsOrdered.put(pos, col);
            }
        }
        List<IColumn> colList = new ArrayList<IColumn>();
        for (int i = 0; i < jaretTable.getTableModel().getColumnCount(); i++) {
            IColumn c = colsOrdered.get(i);
            if (c != null) {
                colList.add(c);
            }
        }
        // add cols that are not in the memento
        for (int i = 0; i < jaretTable.getTableModel().getColumnCount(); i++) {
            IColumn c = jaretTable.getTableModel().getColumn(i);
            if (!colList.contains(c)) {
                colList.add(c);
            }
        }
        
        if (colList.size() > 0) {
            viewstate.setSortedColumns(colList);
        }
    }
}
